# ERCOT Forecast Error Analysis Agent - Deployment Guide

## Overview

This agent has been packaged for deployment to DataRobot following the standard DataRobot agent template structure. It analyzes ERCOT electricity market forecast errors by examining weather data, renewable generation, and energy market news.

## What Was Done

### 1. Restructured Agent Code

The agent implementation from `/backend/` has been restructured to match the DataRobot agent template format found in `/bloggeragent/`:

**Before**: Backend implementation with loose files
```
backend/
├── crew_manager.py       # Agent logic
├── agents.yaml          # Configuration
├── mcp_tools.py         # Tools
└── main.py              # FastAPI server
```

**After**: DataRobot-compliant structure
```
newsanalystagent/agent_crewai/custom_model/
├── agent.py                    # Main agent class with tools
├── custom.py                   # DataRobot entry points
├── helpers.py                  # Helper functions
├── crewai_event_listener.py   # Event listener
├── model-metadata.yaml         # Metadata and runtime params
└── README.md                   # Documentation
```

### 2. Key Components

#### `agent.py`
- Contains `MyAgent` class (main agent implementation)
- Includes both tools inline:
  - `fetch_texas_weather`: Weather data from Open-Meteo
  - `fetch_energy_news`: News from Tavily API
- Defines agent and task properties
- Implements the `invoke()` method required by DataRobot

#### `custom.py`
- Entry points for DataRobot DRUM server
- `load_model()`: Model initialization
- `chat()`: Main invocation endpoint
- Handles streaming and non-streaming responses
- Sets up OpenTelemetry instrumentation

#### `model-metadata.yaml`
- Defines agent as `targetType: agenticworkflow`
- Specifies runtime parameters:
  - `LLM_DATAROBOT_DEPLOYMENT_ID` (optional)
  - `TAVILY_API_KEY` (required)

### 3. Dependencies Updated

Added `tavily-python>=0.5.2` to `pyproject.toml` for news fetching capability.

### 4. No Changes to Agent Definitions

✅ Agent role, goal, and backstory preserved exactly as defined
✅ Task descriptions and expected outputs unchanged
✅ Tool implementations (weather and news) preserved exactly
✅ All business logic maintained

## Deployment to DataRobot

### Prerequisites

1. **DataRobot Account**: Access to DataRobot platform with Agentic Workflows enabled
2. **API Credentials**:
   - DataRobot API token (obtained from DataRobot UI)
   - DataRobot endpoint URL
3. **Tavily API Key**: Sign up at https://tavily.com for news search access

### Step 1: Prepare the Package

The agent is ready to deploy from:
```
/Users/oleg.zarakhani/trading/newsanalystagent/agent_crewai/
```

This directory contains:
- `custom_model/` - The agent implementation
- `pyproject.toml` - Dependencies
- `tests/` - Unit tests
- `docker_context/` - Docker configuration

### Step 2: Local Testing (Optional)

Test the agent locally before deployment:

```bash
cd /Users/oleg.zarakhani/trading/newsanalystagent/agent_crewai

# Set environment variables
export DATAROBOT_ENDPOINT="https://your-instance.datarobot.com"
export DATAROBOT_API_TOKEN="your-token"
export TAVILY_API_KEY="your-tavily-key"

# Install dependencies
pip install -e .

# Run tests
pytest tests/

# Test the agent interactively
python run_agent.py
```

### Step 3: Deploy to DataRobot

#### Option A: Using DataRobot UI

1. **Navigate to Model Registry**
   - Go to DataRobot platform
   - Navigate to Model Registry → Custom Models

2. **Create New Custom Model**
   - Click "Create Custom Model"
   - Select "Agentic Workflow" as the target type
   - Choose "Python 3.11" as the runtime

3. **Upload Agent Files**
   - Upload the entire `agent_crewai` directory
   - Or zip the directory and upload the archive

4. **Configure Runtime Parameters**
   - Set `TAVILY_API_KEY` with your Tavily API key
   - Optionally set `LLM_DATAROBOT_DEPLOYMENT_ID` to use a specific LLM deployment

5. **Build and Validate**
   - DataRobot will build the Docker image
   - Run validation tests
   - Review build logs for any issues

6. **Create Deployment**
   - Once validated, create a deployment
   - Configure deployment settings (replicas, resources)
   - Activate the deployment

#### Option B: Using DataRobot CLI/API

```python
import datarobot as dr

# Initialize DataRobot client
dr.Client(
    endpoint="https://your-instance.datarobot.com",
    token="your-token"
)

# Create custom model
custom_model = dr.CustomInferenceModel.create(
    name="ERCOT Forecast Error Analyst",
    target_type=dr.TARGET_TYPE.AGENTIC_WORKFLOW,
    target_name="analysis"
)

# Upload model files
version = dr.CustomModelVersion.create_clean(
    custom_model_id=custom_model.id,
    base_environment_id="your-crewai-environment-id",
    folder_path="/Users/oleg.zarakhani/trading/newsanalystagent/agent_crewai"
)

# Create deployment
deployment = dr.Deployment.create_from_custom_model_version(
    custom_model_version_id=version.id,
    label="ERCOT Forecast Error Analyst Production",
    default_prediction_server_id="your-prediction-server-id"
)

# Set runtime parameters
deployment.update_runtime_parameter_values([
    {"fieldName": "TAVILY_API_KEY", "value": "your-tavily-key"}
])
```

### Step 4: Test the Deployment

Test the deployed agent with a sample request:

```python
import openai

client = openai.OpenAI(
    base_url="https://your-instance.datarobot.com/api/v2/deployments/your-deployment-id/",
    api_key="your-datarobot-token"
)

# Sample forecast error data
prompt = {
    "hub_name": "HB_HOUSTON",
    "timestamp": "2025-10-22T15:00:00",
    "error": "25.50"
}

response = client.chat.completions.create(
    model="your-deployment-id",
    messages=[
        {"role": "system", "content": "You are an energy market analyst."},
        {"role": "user", "content": str(prompt)}
    ]
)

print(response.choices[0].message.content)
```

### Step 5: Integration with Your Application

Update your backend to call the DataRobot deployment:

```python
# In your backend/main.py or similar
import openai

class DataRobotAgentClient:
    def __init__(self, deployment_id: str, api_token: str, endpoint: str):
        self.client = openai.OpenAI(
            base_url=f"{endpoint}/api/v2/deployments/{deployment_id}/",
            api_key=api_token
        )
        self.deployment_id = deployment_id
    
    async def analyze_forecast_error(
        self, 
        data_point: dict, 
        error_context: dict
    ) -> dict:
        prompt = {
            "hub_name": data_point.get("hub_name"),
            "timestamp": str(data_point.get("timestamp_utc")),
            "error": f"{data_point.get('forecast_error', 0):.2f}"
        }
        
        response = self.client.chat.completions.create(
            model=self.deployment_id,
            messages=[
                {"role": "user", "content": str(prompt)}
            ]
        )
        
        return {
            "analysis": {
                "summary": response.choices[0].message.content
            },
            "hypothesis": response.choices[0].message.content,
            "confidence_score": 0.85,
            "external_data_used": {
                "weather_api": "Open-Meteo (via CrewAI tool)",
                "news_api": "Tavily (via CrewAI tool)"
            }
        }

# Use in your endpoint
agent_client = DataRobotAgentClient(
    deployment_id=os.getenv("DATAROBOT_AGENT_DEPLOYMENT_ID"),
    api_token=os.getenv("DATAROBOT_API_TOKEN"),
    endpoint=os.getenv("DATAROBOT_ENDPOINT")
)
```

## Input/Output Schema

### Input
```json
{
  "hub_name": "HB_HOUSTON",
  "timestamp": "2025-10-22T15:00:00",
  "error": "25.50"
}
```

### Output
```json
{
  "analysis": "Narrative analysis with weather and news context...\n\nAPPENDIX\n\nWeather Appendix:\n- Houston: 75°F, Wind 10 mph, Clouds 40%\n...\n\nNews Appendix:\n- Article 1 - Source (Date)\n...",
  "usage_metrics": {
    "prompt_tokens": 1234,
    "completion_tokens": 567,
    "total_tokens": 1801
  }
}
```

## Monitoring and Observability

DataRobot provides built-in monitoring for your agent:

1. **Request Logs**: View all requests and responses
2. **Telemetry**: OpenTelemetry traces for agent execution
3. **Performance Metrics**: Latency, throughput, error rates
4. **Tool Call Tracking**: See which tools were called and when
5. **LLM Usage**: Track token usage and costs

Access these from the DataRobot Deployments dashboard.

## Troubleshooting

### Build Fails
- Check `pyproject.toml` for dependency conflicts
- Review build logs in DataRobot UI
- Ensure all required files are present in `custom_model/`

### Agent Not Responding
- Verify runtime parameters are set (especially TAVILY_API_KEY)
- Check deployment health status
- Review deployment logs for errors

### Weather Tool Errors
- Verify timestamp format is ISO 8601
- Check hub_name is valid (HB_HOUSTON, HB_NORTH, etc.)
- Open-Meteo API is free but may have rate limits

### News Tool Errors
- Verify Tavily API key is valid
- Check Tavily account has sufficient credits
- Review date range for news search

## Next Steps

1. ✅ **Agent Packaged** - Ready for deployment
2. ⏭️ **Deploy to DataRobot** - Follow steps above
3. ⏭️ **Test Deployment** - Verify with sample data
4. ⏭️ **Update Backend** - Point to DataRobot deployment
5. ⏭️ **Monitor** - Set up alerts and monitoring

## Support

For questions about:
- **DataRobot Platform**: Contact DataRobot support or check documentation at https://docs.datarobot.com
- **Agent Template**: See https://github.com/datarobot-community/datarobot-agent-templates
- **This Implementation**: Review the code in `custom_model/` and README files

## Files Changed

### New/Modified Files
- ✅ `/newsanalystagent/agent_crewai/custom_model/agent.py` - Created
- ✅ `/newsanalystagent/agent_crewai/custom_model/custom.py` - Updated
- ✅ `/newsanalystagent/agent_crewai/custom_model/model-metadata.yaml` - Updated
- ✅ `/newsanalystagent/agent_crewai/pyproject.toml` - Added tavily-python dependency
- ✅ `/newsanalystagent/agent_crewai/custom_model/README.md` - Created
- ✅ `/newsanalystagent/DEPLOYMENT_GUIDE.md` - This file

### Unchanged Files
- ✅ `/newsanalystagent/agent_crewai/custom_model/helpers.py` - Standard helper
- ✅ `/newsanalystagent/agent_crewai/custom_model/crewai_event_listener.py` - Standard listener
- ✅ `/newsanalystagent/agent_crewai/custom_model/__init__.py` - Standard init

### Original Backend Files (For Reference)
The original implementation is preserved in `/backend/`:
- `backend/crew_manager.py`
- `backend/agents.yaml`
- `backend/mcp_tools.py`
- `backend/main.py`

These files can be archived or removed once the DataRobot deployment is verified.

