import os
from typing import Iterator

from datarobot_drum import RuntimeParameters
from openai import AzureOpenAI, NotFoundError
from openai.types.chat import (
    ChatCompletion,
    ChatCompletionChunk,
    CompletionCreateParams,
)


def load_model(*args, **kwargs) -> AzureOpenAI:
    try:
        api_key = RuntimeParameters.get("OPENAI_API_KEY")["apiToken"]
        api_base = RuntimeParameters.get("OPENAI_API_BASE")
        api_version = RuntimeParameters.get("OPENAI_API_VERSION")
        model_name = RuntimeParameters.get("OPENAI_API_DEPLOYMENT_ID")
    except Exception as e:
        print(f"Error loading model. Trying local credentials..: {e}")
        api_key = os.getenv("OPENAI_API_KEY")
        api_base = os.getenv("OPENAI_API_BASE")
        api_version = os.getenv("OPENAI_API_VERSION")
        model_name = os.getenv("OPENAI_API_DEPLOYMENT_ID")

    return (
        AzureOpenAI(
            api_key=api_key,
            azure_endpoint=api_base,
            api_version=api_version,
        ),
        model_name,
    )


def chat(
    completion_create_params: CompletionCreateParams, model: tuple[AzureOpenAI, str]
) -> ChatCompletion | Iterator[ChatCompletionChunk]:
    """Chat Hook compatible with ChatCompletion OpenAI Specification.

    Accepts a user defined and defaults to runtime parameter set model

    Parameters
    ----------
    completion_create_params : CompletionCreateParams
        object that holds all the parameters needed to create the chat completion.
    model : tuple[AzureOpenAI, str]
        The model object from openai and the name.  this is injected output from load_model

    Returns
    -------
    ChatCompletion
        the completion object with generated choices.
    """
    client, default_model_name = model
    try:
        return client.chat.completions.create(**completion_create_params)
    except (NotFoundError, TypeError):
        completion_create_params["model"] = default_model_name
        return client.chat.completions.create(**completion_create_params)
